const database = require('./database');
const translator = require('./translator');
const config = require('../config/config');
const countryLanguages = require('./countryLanguages');
const departmentCommands = require('./departmentCommands');
const fs = require('fs');
const path = require('path');

class AdminCommands {
  constructor() {
    this.adminStates = new Map(); // Para fluxos de comandos admin
  }

  /**
   * Check if user is admin
   */
  isAdmin(phone) {
    return config.admins.includes(phone);
  }

  /**
   * Check if user is agent
   */
  isAgent(phone) {
    const agent = database.getAgent(phone);
    return agent && agent.status === 'active';
  }

  /**
   * Get admin/agent language
   */
  getAdminLanguage(phone) {
    console.log(`🔍 Getting language for admin ${phone}`);
    
    // Try to get from agents table first
    const agent = database.getAgent(phone);
    if (agent && agent.language) {
      console.log(`   ✅ Found in agents table: ${agent.language}`);
      return this.validateLanguageCode(agent.language);
    }
    
    // Try to get from clients table
    const client = database.getClient(phone);
    if (client && client.language && client.language_confirmed === 1) {
      console.log(`   ✅ Found in clients table: ${client.language}`);
      return this.validateLanguageCode(client.language);
    }
    
    // Default to English
    console.log(`   ⚠️ No language found, defaulting to 'en'`);
    return 'en';
  }

  /**
   * Validate and sanitize language code
   */
  validateLanguageCode(lang) {
    if (!lang || typeof lang !== 'string') {
      return 'en';
    }
    
    // Clean the language code
    lang = lang.toLowerCase().trim();
    
    // Check if it's a valid 2-5 character code
    if (lang.length < 2 || lang.length > 5) {
      return 'en';
    }
    
    // List of common valid language codes
    const validCodes = ['en', 'pt', 'es', 'fr', 'de', 'it', 'zh', 'ja', 'ko', 'ru', 'ar', 'hi', 'nl', 'sv', 'no', 'da', 'fi', 'pl', 'tr', 'th', 'vi', 'id', 'ms', 'tl'];
    
    // If it's in the list, return it
    if (validCodes.includes(lang)) {
      return lang;
    }
    
    // If it looks like a valid code (2 letters), allow it
    if (/^[a-z]{2}$/.test(lang)) {
      return lang;
    }
    
    // Otherwise, default to English
    console.warn(`⚠️ Invalid language code "${lang}", defaulting to 'en'`);
    return 'en';
  }

  /**
   * Translate message for specific admin
   */
  async translateForAdmin(textEN, phone) {
    const adminLang = this.getAdminLanguage(phone);
    
    console.log(`🌐 Translating admin message from 'en' to '${adminLang}'`);
    
    // If admin language is English, return directly
    if (adminLang === 'en') {
      console.log(`   ⏭️ Same language, skipping translation`);
      return textEN;
    }
    
    // Validate language code before translating
    if (!adminLang || adminLang.length < 2) {
      console.error(`   ❌ Invalid language code "${adminLang}", using English`);
      return textEN;
    }
    
    try {
      const result = await translator.translate(textEN, adminLang, 'en');
      console.log(`   ✅ Translated successfully`);
      return result.translatedText;
    } catch (error) {
      console.error(`   ❌ Translation error:`, error.message);
      console.error(`   Using original text in English as fallback`);
      return textEN; // Return in EN as fallback
    }
  }

  /**
   * Process admin command
   */
  async processAdminCommand(phone, command, args) {
    if (!this.isAdmin(phone)) {
      const errorEN = '❌ You do not have admin permissions.';
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    // Check if admin is in a flow
    if (this.adminStates.has(phone)) {
      return await this.processAdminFlow(phone, command);
    }

    // Process admin commands
    switch (command) {
      case '/add agent':
      case '/addagent':
        return await this.handleAddAgent(phone);

      case '/remove agent':
      case '/removeagent':
        return await this.handleRemoveAgent(phone, args[0]);

      case '/edit agent':
      case '/editagent':
        return await this.handleEditAgent(phone, args[0]);

      case '/list agents':
      case '/listagents':
      case '/agents':
        return await this.handleListAgents(phone);

      case '/agent stats':
      case '/agentstats':
        return await this.handleAgentStats(phone, args[0]);

      case '/all stats':
      case '/allstats':
        return await this.handleAllStats(phone);

      case '/change hours':
      case '/changehours':
        return await this.handleChangeHours(phone);

      case '/change bot name':
      case '/changebotname':
        return await this.handleChangeBotName(phone);

      case '/change company':
      case '/changecompany':
        return await this.handleChangeCompany(phone);

      case '/set agent names':
      case '/setagentnames':
        return await this.handleSetAgentNames(phone, args[0]);

      // Department commands with multi-word syntax
      case '/toggle':
        if (args[0] === 'departments') {
          return await departmentCommands.toggleDepartments(phone);
        }
        return null;
      
      case '/toggledepartments':
        return await departmentCommands.toggleDepartments(phone);

      case '/create':
        if (args[0] === 'department') {
          return await departmentCommands.createDepartment(phone, args.slice(1));
        }
        return null;
      
      case '/createdepartment':
        return await departmentCommands.createDepartment(phone, args);

      case '/list':
        if (args[0] === 'departments') {
          return await departmentCommands.listDepartments(phone);
        }
        return null;
      
      case '/listdepartments':
      case '/departments':
        return await departmentCommands.listDepartments(phone);

      case '/department':
        if (args[0] === 'info' && args[1]) {
          return await departmentCommands.getDepartmentInfo(phone, args[1]);
        }
        return null;

      case '/add':
        if (args[0] === 'agent' && args[1] === 'to' && args[2] === 'department') {
          return await departmentCommands.addAgentToDepartment(phone, args.slice(3));
        }
        return null;

      case '/remove':
        if (args[0] === 'agent' && args[1] === 'from' && args[2] === 'department') {
          return await departmentCommands.removeAgentFromDepartment(phone, args.slice(3));
        }
        return null;

      case '/set':
        if (args[0] === 'general' && args[1] === 'department' && args[2]) {
          return await departmentCommands.setGeneralDepartment(phone, args[2]);
        }
        return null;

      case '/edit':
        if (args[0] === 'department' && args[1]) {
          return await departmentCommands.editDepartment(phone, args[1]);
        }
        return null;

      case '/delete':
        if (args[0] === 'department' && args[1]) {
          return await departmentCommands.deleteDepartment(phone, args[1]);
        }
        return null;

      case '/restart':
      case '/restart bot':
        return await this.handleRestartBot(phone);

      case '/admin help':
      case '/adminhelp':
        return await this.handleAdminHelp(phone);

      default:
        return null; // Not an admin command
    }
  }

  /**
   * Add new agent - start flow
   */
  async handleAddAgent(phone) {
    this.adminStates.set(phone, {
      flow: 'add_agent',
      step: 'phone',
      data: {}
    });

    const messageEN = `➕ *Add New Agent*

📞 Please enter the agent's phone number (with country code):

Example: 258840000000

Type "cancel" to cancel.`;

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: 'start_add_agent'
    };
  }

  /**
   * Remove agent
   */
  async handleRemoveAgent(phone, agentPhone) {
    if (!agentPhone) {
      const errorEN = `❌ Invalid format.

Usage: /remove agent [phone]
Example: /remove agent 258840000000`;
      
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    const agent = database.getAgent(agentPhone);
    if (!agent) {
      const errorEN = '❌ Agent not found.';
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    database.removeAgent(agentPhone);

    const successEN = `✅ *Agent Removed*

👤 Name: ${agent.name}
📞 Phone: ${agent.phone}

The agent no longer has access to the system.`;

    const success = await this.translateForAdmin(successEN, phone);
    return {
      success: true,
      response: success,
      action: 'agent_removed'
    };
  }

  /**
   * Edit agent - start flow
   */
  async handleEditAgent(phone, agentPhone) {
    if (!agentPhone) {
      const errorEN = `❌ Invalid format.

Usage: /edit agent [phone]
Example: /edit agent 258840000000`;
      
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    const agent = database.getAgent(agentPhone);
    if (!agent) {
      const errorEN = '❌ Agent not found.';
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    this.adminStates.set(phone, {
      flow: 'edit_agent',
      step: 'menu',
      data: { agentPhone }
    });

    const messageEN = `✏️ *Edit Agent*

👤 Name: ${agent.name}
📞 Phone: ${agent.phone}
🌍 Language: ${translator.getLanguageName(agent.language)}

What do you want to edit?

1️⃣ Name
2️⃣ Language

Type the number or "cancel" to cancel.`;

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: 'start_edit_agent'
    };
  }

  /**
   * List all agents
   */
  async handleListAgents(phone) {
    const agents = database.getAllAgents();

    if (agents.length === 0) {
      const messageEN = '📋 No agents registered.';
      const message = await this.translateForAdmin(messageEN, phone);
      return {
        success: true,
        response: message,
        action: null
      };
    }

    let messageEN = `👥 *Registered Agents*\n\n`;

    for (const agent of agents) {
      messageEN += `📞 ${agent.phone}\n`;
      messageEN += `👤 Name: ${agent.name}\n`;
      messageEN += `🌍 Language: ${translator.getLanguageName(agent.language)}\n`;
      messageEN += `📊 Tickets: ${agent.total_tickets} total\n`;
      messageEN += `━━━━━━━━━━━━━━\n\n`;
    }

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: null
    };
  }

  /**
   * Get agent statistics
   */
  async handleAgentStats(phone, agentPhone) {
    if (!agentPhone) {
      const errorEN = `❌ Invalid format.

Usage: /agent stats [phone]
Example: /agent stats 258840000000`;
      
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    const stats = database.getAgentStats(agentPhone);
    if (!stats) {
      const errorEN = '❌ Agent not found.';
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    const messageEN = `📊 *Agent Statistics*

👤 Name: ${stats.name}
📞 Phone: ${stats.phone}
🌍 Language: ${translator.getLanguageName(stats.language)}

━━━━━━━━━━━━━━━━━━━━
📈 *Tickets Handled*
━━━━━━━━━━━━━━━━━━━━

📅 Today: ${stats.today}
📆 This Week: ${stats.week}
📆 This Month: ${stats.month}
📊 Total: ${stats.total}`;

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: null
    };
  }

  /**
   * Get all agents statistics
   */
  async handleAllStats(phone) {
    const allStats = database.getAllAgentsStats();

    if (allStats.length === 0) {
      const messageEN = '📋 No agents registered.';
      const message = await this.translateForAdmin(messageEN, phone);
      return {
        success: true,
        response: message,
        action: null
      };
    }

    let messageEN = `📊 *All Agents Statistics*\n\n`;

    for (const stats of allStats) {
      messageEN += `👤 ${stats.name} (${stats.phone})\n`;
      messageEN += `   Today: ${stats.today} | Week: ${stats.week} | Month: ${stats.month}\n`;
      messageEN += `   Total: ${stats.total}\n`;
      messageEN += `━━━━━━━━━━━━━━\n`;
    }

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: null
    };
  }

  /**
   * Change business hours - start flow
   */
  async handleChangeHours(phone) {
    this.adminStates.set(phone, {
      flow: 'change_hours',
      step: 'start',
      data: {}
    });

    const messageEN = `🕐 *Change Business Hours*

Current hours: ${config.businessHours.start} to ${config.businessHours.end}
Timezone: ${config.businessHours.timezone}

Send new hours in format: HH:MM-HH:MM

Example: 09:00-18:00

Type "cancel" to cancel.`;

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: 'start_change_hours'
    };
  }

  /**
   * Change bot name - start flow
   */
  async handleChangeBotName(phone) {
    this.adminStates.set(phone, {
      flow: 'change_bot_name',
      step: 'name',
      data: {}
    });

    const messageEN = `🤖 *Change Bot Name*

Current name: ${config.bot.name}

Send the new bot name:

Type "cancel" to cancel.`;

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: 'start_change_bot_name'
    };
  }

  /**
   * Change company name - start flow
   */
  async handleChangeCompany(phone) {
    this.adminStates.set(phone, {
      flow: 'change_company',
      step: 'input'
    });

    const messageEN = `🏢 *Change Company Name*

Current name: ${config.bot.company}

Send the new company name:

Type "cancel" to cancel.`;

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: 'start_change_company'
    };
  }

  /**
   * Set agent names mode (real or random) - GLOBAL setting
   */
  async handleSetAgentNames(phone, mode) {
    if (!mode || (mode !== 'real' && mode !== 'random')) {
      const errorEN = `❌ Invalid format.

Usage: /set agent names [real|random]

Examples:
• /set agent names real - Use real agent names
• /set agent names random - Use random fictitious names`;
      
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    const useRealNames = mode === 'real' ? '1' : '0';
    
    // Save to settings
    database.saveSetting('agent_names_mode', useRealNames);

    const modeText = mode === 'real' ? 'Real Names' : 'Random Names';
    const successEN = `✅ *Agent Names Mode Updated!*

👥 Mode: ${modeText}

${mode === 'real' ? '✓ Clients will see the real names of agents' : '✓ Clients will see random professional names'}

This applies to all current and future agents.`;

    const success = await this.translateForAdmin(successEN, phone);
    return {
      success: true,
      response: success,
      action: 'names_mode_updated'
    };
  }

  /**
   * Restart bot
   */
  async handleRestartBot(phone) {
    const confirmEN = `🔄 *Restarting Bot...*

The bot will restart in 3 seconds.

⚠️ All connections will be temporarily interrupted.
✅ The bot will reconnect automatically.

Please wait...`;

    const confirm = await this.translateForAdmin(confirmEN, phone);
    
    // Send confirmation message
    setTimeout(() => {
      console.log('🔄 Bot restart requested by admin:', phone);
      console.log('⏳ Restarting in 3 seconds...');
      
      // Exit process - PM2/systemd will restart it automatically
      setTimeout(() => {
        process.exit(0);
      }, 3000);
    }, 100);

    return {
      success: true,
      response: confirm,
      action: 'restart_bot'
    };
  }

  /**
   * Admin help menu
   */
  async handleAdminHelp(phone) {
    const messageEN = `🔐 *Admin Commands*

━━━━━━━━━━━━━━━━━━━━
👥 *Agent Management*
━━━━━━━━━━━━━━━━━━━━

• */add agent* - Add new agent
• */remove agent [phone]* - Remove agent
• */edit agent [phone]* - Edit agent
• */list agents* - List all agents

━━━━━━━━━━━━━━━━━━━━
🏢 *Department System*
━━━━━━━━━━━━━━━━━━━━

• */toggle departments* - Enable/Disable departments
• */create department [name]* - Create department
• */list departments* - List departments
• */department info [ID]* - Department details
• */add agent to department* - Assign agent
• */remove agent from department* - Remove agent
• */set general department [ID]* - Set general dept
• */edit department [ID]* - Edit department
• */delete department [ID]* - Delete department

━━━━━━━━━━━━━━━━━━━━
📊 *Statistics*
━━━━━━━━━━━━━━━━━━━━

• */agent stats [phone]* - Agent statistics
• */all stats* - All agents statistics

━━━━━━━━━━━━━━━━━━━━
⚙️ *System Configuration*
━━━━━━━━━━━━━━━━━━━━

• */change hours* - Change business hours
• */change bot name* - Change bot name
• */change company* - Change company name
• */set agent names [real|random]* - Agent names mode
• */restart* - Restart the bot

━━━━━━━━━━━━━━━━━━━━

💡 Type */admin help* to see this menu again.`;

    const message = await this.translateForAdmin(messageEN, phone);
    return {
      success: true,
      response: message,
      action: null
    };
  }

  /**
   * Process admin flow
   */
  async processAdminFlow(phone, message) {
    const state = this.adminStates.get(phone);
    if (!state) return null;

    console.log(`🔍 Admin flow state:`, state);

    // Check for cancel
    if (message.toLowerCase() === 'cancel' || message.toLowerCase() === 'cancelar') {
      this.adminStates.delete(phone);
      
      const cancelEN = '❌ Operation cancelled.';
      const cancel = await this.translateForAdmin(cancelEN, phone);
      return {
        success: true,
        response: cancel,
        action: 'cancelled'
      };
    }

    switch (state.flow) {
      case 'add_agent':
        return await this.processAddAgentFlow(phone, message, state);

      case 'edit_agent':
        return await this.processEditAgentFlow(phone, message, state);

      case 'change_hours':
        return await this.processChangeHoursFlow(phone, message, state);

      case 'change_bot_name':
        return await this.processChangeBotNameFlow(phone, message, state);

      case 'change_company':
        return await this.processChangeCompanyFlow(phone, message, state);

      // Department flows
      case 'awaiting_agent_phone_for_department':
        return await this.processAgentPhoneForDepartment(phone, message);

      case 'awaiting_department_selection':
        return await this.processDepartmentSelection(phone, message, state);

      case 'awaiting_agent_phone_for_removal':
        return await this.processAgentPhoneForRemoval(phone, message);

      case 'awaiting_department_removal':
        return await this.processDepartmentRemovalSelection(phone, message, state);

      case 'awaiting_department_edit':
        return await this.processDepartmentNameEdit(phone, message, state);

      default:
        this.adminStates.delete(phone);
        return null;
    }
  }

  /**
   * Process add agent flow
   */
  async processAddAgentFlow(phone, message, state) {
    switch (state.step) {
      case 'phone':
        // Validate phone number
        const agentPhone = message.trim();
        if (!/^\d{10,15}$/.test(agentPhone)) {
          const errorEN = '❌ Invalid phone number. Please enter only numbers (10-15 digits).';
          const error = await this.translateForAdmin(errorEN, phone);
          return {
            success: false,
            response: error,
            action: null
          };
        }

        // Check if agent already exists
        if (database.getAgent(agentPhone)) {
          const errorEN = '❌ This agent is already registered.';
          const error = await this.translateForAdmin(errorEN, phone);
          return {
            success: false,
            response: error,
            action: null
          };
        }
        state.data.phone = agentPhone;
        state.step = 'name';
        this.adminStates.set(phone, state);

        const nameEN = `✅ Phone confirmed: ${agentPhone}

Now send the agent's name:`;
        const nameMsg = await this.translateForAdmin(nameEN, phone);
        return {
          success: true,
          response: nameMsg,
          action: 'phone_confirmed'
        };

      case 'name':
        state.data.name = message.trim();
        state.step = 'language';
        this.adminStates.set(phone, state);

        // Use country-json for comprehensive language list
        const langEN = `✅ Name confirmed: ${message.trim()}

${countryLanguages.getPopularLanguagesFormatted()}`;
        const langMsg = await this.translateForAdmin(langEN, phone);
        return {
          success: true,
          response: langMsg,
          action: 'name_confirmed'
        };

      case 'language':
        // Parse language selection using countryLanguages
        let lang = countryLanguages.parseLanguageSelection(message);
        
        if (!lang || !countryLanguages.isValidLanguageCode(lang)) {
          const errorEN = '❌ Invalid language selection. Please choose a number (1-12) or enter a valid ISO language code (e.g., en, pt, es)';
          const error = await this.translateForAdmin(errorEN, phone);
          return {
            success: false,
            response: error,
            action: null
          };
        }

        // Add agent to database (without name mode - it's now global)
        try {
          database.addAgent(state.data.phone, state.data.name, lang, phone);
          this.adminStates.delete(phone);

          const successEN = `✅ *Agent Added Successfully!*

👤 Name: ${state.data.name}
📞 Phone: ${state.data.phone}
🌍 Language: ${countryLanguages.getLanguageName(lang)}

The agent can now receive and handle tickets!`;

          const success = await this.translateForAdmin(successEN, phone);
          return {
            success: true,
            response: success,
            action: 'agent_added'
          };
        } catch (error) {
          this.adminStates.delete(phone);
          
          const errorEN = `❌ Error adding agent: ${error.message}`;
          const errorMsg = await this.translateForAdmin(errorEN, phone);
          return {
            success: false,
            response: errorMsg,
            action: null
          };
        }

      default:
        this.adminStates.delete(phone);
        return null;
    }
  }

  /**
   * Process edit agent flow
   */
  async processEditAgentFlow(phone, message, state) {
    switch (state.step) {
      case 'menu':
        if (message === '1') {
          state.step = 'name';
          this.adminStates.set(phone, state);

          const nameEN = 'Send the new name for the agent:';
          const nameMsg = await this.translateForAdmin(nameEN, phone);
          return {
            success: true,
            response: nameMsg,
            action: 'edit_name'
          };
        } else if (message === '2') {
          state.step = 'language';
          this.adminStates.set(phone, state);

          const langEN = `${countryLanguages.getPopularLanguagesFormatted()}`;
          const langMsg = await this.translateForAdmin(langEN, phone);
          return {
            success: true,
            response: langMsg,
            action: 'edit_language'
          };
        } else {
          const errorEN = '❌ Invalid option. Type 1 or 2.';
          const error = await this.translateForAdmin(errorEN, phone);
          return {
            success: false,
            response: error,
            action: null
          };
        }

      case 'name':
        database.updateAgent(state.data.agentPhone, { name: message.trim() });
        this.adminStates.delete(phone);

        const successEN = `✅ Agent name updated to: ${message.trim()}`;
        const success = await this.translateForAdmin(successEN, phone);
        return {
          success: true,
          response: success,
          action: 'agent_updated'
        };

      case 'language':
        let lang = countryLanguages.parseLanguageSelection(message);
        
        if (!lang || !countryLanguages.isValidLanguageCode(lang)) {
          const errorEN = '❌ Invalid language selection. Please choose a number (1-12) or enter a valid ISO language code (e.g., en, pt, es)';
          const error = await this.translateForAdmin(errorEN, phone);
          return {
            success: false,
            response: error,
            action: null
          };
        }

        database.updateAgent(state.data.agentPhone, { language: lang });
        this.adminStates.delete(phone);

        const langSuccessEN = `✅ Agent language updated to: ${countryLanguages.getLanguageName(lang)}`;
        const langSuccess = await this.translateForAdmin(langSuccessEN, phone);
        return {
          success: true,
          response: langSuccess,
          action: 'agent_updated'
        };

      default:
        this.adminStates.delete(phone);
        return null;
    }
  }

  /**
   * Process change hours flow
   */
  async processChangeHoursFlow(phone, message, state) {
    // Validate format HH:MM-HH:MM
    const match = message.match(/^(\d{2}):(\d{2})-(\d{2}):(\d{2})$/);
    if (!match) {
      const errorEN = '❌ Invalid format. Use: HH:MM-HH:MM\nExample: 09:00-18:00';
      const error = await this.translateForAdmin(errorEN, phone);
      return {
        success: false,
        response: error,
        action: null
      };
    }

    const [_, startH, startM, endH, endM] = match;
    const start = `${startH}:${startM}`;
    const end = `${endH}:${endM}`;

    // Update .env file
    this.updateEnvFile('BUSINESS_HOURS_START', start);
    this.updateEnvFile('BUSINESS_HOURS_END', end);

    this.adminStates.delete(phone);

    const successEN = `✅ *Business Hours Updated!*

⏰ New hours: ${start} to ${end}

⚠️ Restart the bot for changes to take effect.`;

    const success = await this.translateForAdmin(successEN, phone);
    return {
      success: true,
      response: success,
      action: 'hours_updated'
    };
  }

  /**
   * Process change bot name flow
   */
  async processChangeBotNameFlow(phone, message, state) {
    const newName = message.trim();
    
    // Update .env file
    this.updateEnvFile('BOT_NAME', newName);

    this.adminStates.delete(phone);

    const successEN = `✅ *Bot Name Updated!*

🤖 New name: ${newName}

⚠️ Restart the bot for changes to take effect.`;

    const success = await this.translateForAdmin(successEN, phone);
    return {
      success: true,
      response: success,
      action: 'bot_name_updated'
    };
  }

  /**
   * Process change company flow
   */
  async processChangeCompanyFlow(phone, message, state) {
    const newCompany = message.trim();
    
    // Update .env file
    this.updateEnvFile('COMPANY_NAME', newCompany);

    this.adminStates.delete(phone);

    const successEN = `✅ *Company Name Updated!*

🏢 New name: ${newCompany}

⚠️ Restart the bot for changes to take effect.`;

    const success = await this.translateForAdmin(successEN, phone);
    return {
      success: true,
      response: success,
      action: 'company_updated'
    };
  }

  /**
   * Update .env file
   */
  updateEnvFile(key, value) {
    try {
      const envPath = path.join(__dirname, '../.env');
      let envContent = fs.readFileSync(envPath, 'utf8');

      // Update or add the key
      const regex = new RegExp(`^${key}=.*$`, 'm');
      if (regex.test(envContent)) {
        envContent = envContent.replace(regex, `${key}=${value}`);
      } else {
        envContent += `\n${key}=${value}`;
      }

      fs.writeFileSync(envPath, envContent, 'utf8');
    } catch (error) {
      console.error(`❌ Error updating .env file:`, error.message);
    }
  }

  // ==================== DEPARTMENT FLOW PROCESSORS ====================

  /**
   * Process agent phone for department assignment
   */
  async processAgentPhoneForDepartment(phone, agentPhone) {
    const result = await departmentCommands.processAgentPhoneForDepartment(phone, agentPhone);
    
    if (result.success && result.action === 'awaiting_department_selection') {
      this.adminStates.set(phone, {
        flow: 'awaiting_department_selection',
        agentPhone: result.agentPhone,
        departments: result.departments
      });
    } else {
      this.adminStates.delete(phone);
    }
    
    return result;
  }

  /**
   * Process department selection for agent assignment
   */
  async processDepartmentSelection(phone, message, state) {
    const selection = parseInt(message.trim());
    
    if (isNaN(selection) || selection < 1 || selection > state.departments.length) {
      return {
        success: false,
        response: `❌ Invalid selection. Please send a number between 1 and ${state.departments.length}`
      };
    }

    const department = state.departments[selection - 1];
    const result = await departmentCommands.confirmAgentToDepartment(phone, state.agentPhone, department.id);
    
    this.adminStates.delete(phone);
    return result;
  }

  /**
   * Process agent phone for removal
   */
  async processAgentPhoneForRemoval(phone, agentPhone) {
    const result = await departmentCommands.processAgentRemovalFromDepartment(phone, agentPhone);
    
    if (result.success && result.action === 'awaiting_department_removal') {
      this.adminStates.set(phone, {
        flow: 'awaiting_department_removal',
        agentPhone: result.agentPhone,
        departments: result.departments
      });
    } else {
      this.adminStates.delete(phone);
    }
    
    return result;
  }

  /**
   * Process department removal selection
   */
  async processDepartmentRemovalSelection(phone, message, state) {
    const selection = parseInt(message.trim());
    
    if (isNaN(selection) || selection < 1 || selection > state.departments.length) {
      return {
        success: false,
        response: `❌ Invalid selection. Please send a number between 1 and ${state.departments.length}`
      };
    }

    const department = state.departments[selection - 1];
    const result = await departmentCommands.confirmAgentRemoval(phone, state.agentPhone, department.id);
    
    this.adminStates.delete(phone);
    return result;
  }

  /**
   * Process department name edit
   */
  async processDepartmentNameEdit(phone, message, state) {
    const result = await departmentCommands.processDepartmentEdit(phone, state.departmentId, message.trim());
    
    this.adminStates.delete(phone);
    return result;
  }
}

module.exports = new AdminCommands();
